<?php

/**
 * @file ReviewCreditsReportPlugin.inc.php
 *
 * Copyright (c) 2014-2020 Simon Fraser University
 * Copyright (c) 2003-2020 John Willinsky
 * Distributed under the GNU GPL v3. For full terms see the file docs/COPYING.
 *
 * @class ReviewReportPlugin
 * @ingroup plugins_reports_review
 * @see ReviewReportDAO
 *
 * @brief Review report plugin
 */

import( 'lib.pkp.classes.plugins.ReportPlugin' );

class ReviewerCreditsReportPlugin extends ReportPlugin {
	/**
	 * ReviewerCreditsReportPlugin constructor.
	 */
	public function __construct() {

		if ( Config::getVar( 'general', 'installed' ) ) {
			require_once( 'plugins/generic/reviewerCredits/ReviewerCreditsReportDAO.inc.php' );
			$reviewReportDAO = new ReviewerCreditsReportDAO();
			DAORegistry::registerDAO( 'ReviewerCreditsReportDAO', $reviewReportDAO );
		}
		$this->addLocaleData();

		return true;
	}

	/**
	 * @copydoc Plugin::getName()
	 */
	function getName() {
		return 'ReviewerCreditsReportPlugin';
	}

	/**
	 * @copydoc Plugin::getDisplayName()
	 */
	function getDisplayName() {
		return __( 'plugins.generic.reviewerCredits.report.displayName' );
	}

	/**
	 * @copydoc Plugin::getDescription()
	 */
	function getDescription() {
		return __( 'plugins.generic.reviewerCredits.report.description' );
	}

	/**
	 * @copydoc ReportPlugin::display()
	 */
	function display( $args, $request ) {
		$context = $request->getContext();

		header( 'content-type: text/comma-separated-values' );
		header( 'content-disposition: attachment; filename=ReviewerCredits-reviews-' . date( 'Ymd' ) . '.csv' );
		AppLocale::requireComponents( LOCALE_COMPONENT_PKP_SUBMISSION );

		$reviewReportDao = DAORegistry::getDAO( 'ReviewerCreditsReportDAO' );
		list( $commentsIterator, $reviewsIterator ) = $reviewReportDao->getReviewReport( $context->getId() );

		$comments = [];
		foreach ( $commentsIterator as $row ) {
			$comments[ $row->submission_id ][ $row->author_id ][ $row->review_id ] = $row->comments;
		}

		import( 'lib.pkp.classes.submission.reviewAssignment.ReviewAssignment' );
		$recommendations = [
			SUBMISSION_REVIEWER_RECOMMENDATION_ACCEPT             => 'reviewer.article.decision.accept',
			SUBMISSION_REVIEWER_RECOMMENDATION_PENDING_REVISIONS  => 'reviewer.article.decision.pendingRevisions',
			SUBMISSION_REVIEWER_RECOMMENDATION_RESUBMIT_HERE      => 'reviewer.article.decision.resubmitHere',
			SUBMISSION_REVIEWER_RECOMMENDATION_RESUBMIT_ELSEWHERE => 'reviewer.article.decision.resubmitElsewhere',
			SUBMISSION_REVIEWER_RECOMMENDATION_DECLINE            => 'reviewer.article.decision.decline',
			SUBMISSION_REVIEWER_RECOMMENDATION_SEE_COMMENTS       => 'reviewer.article.decision.seeComments'
		];

		$columns = [
			'journal_title'  => "Journal Title",
			'journal_pissn'  => "Journal p-ISSN",
			'journal_eissn'  => "Journal e-ISSN",
			'user_given'     => "Reviewer First Name",
			'user_family'    => "Reviewer Last Name",
			'reviewer_email' => "E-mail Address",
			'orcid'          => "ORCID iD or ORCID URL",
			'manuscript'     => "Manuscript ID",
			'revision'       => "Revision Number",
			'date_due'       => "Date Review Due",
			'date_completed' => "Date Review Complete",
			'open_pr'        => "The peer reviewer identity is disclosed (OPEN PEER REVIEW)",
			'doi'            => "Publication DOI or DOI URL"
		];

//		$f3 = ( new ReviewerCreditsPlugin() )->getSetting( $context->getId(), 'reviewerCreditsJournalF3' );
//		if ( ! empty( $f3 ) && $f3 ) {
//			$columns = array_merge( $columns,
//				[
//					'date_accepted'  => "Acceptance Date",
//					'recommendation' => "Recommendation",
//					'comment_length' => "Comment Length"
//				] );
//		}

		$fp = fopen( 'php://output', 'wt' );
		//Add BOM (byte order mark) to fix UTF-8 in Excel
		fprintf( $fp, chr( 0xEF ) . chr( 0xBB ) . chr( 0xBF ) );
		fputcsv( $fp, array_values( $columns ) );

		$reviewAssignmentDao   = DAORegistry::getDAO( 'ReviewAssignmentDAO' );
		$userDao               = DAORegistry::getDAO( 'UserDAO' );

		$journal      = $request->getJournal();
		$journalTitle = $journal->getLocalizedPageHeaderTitle();
		$journalPIssn = $journal->getData( 'printIssn' );
		$journalEIssn = $journal->getData( 'onlineIssn' );

		foreach ( $reviewsIterator as $row ) {
			if ( substr( $row->date_response_due, 11 ) === '00:00:00' ) {
				$row->date_response_due = substr( $row->date_response_due, 0, 11 ) . '23:59:59';
			}
			if ( substr( $row->date_due, 11 ) === '00:00:00' ) {
				$row->date_due = substr( $row->date_due, 0, 11 ) . '23:59:59';
			}
			$reviewId = $row->review_id;

			$reviewAssignment = $reviewAssignmentDao->getById( $reviewId );

			$reviewer = $userDao->getById( $reviewAssignment->getReviewerId() );

			$manuscriptId = 'PR' . $row->review_id . '-S' . $reviewAssignment->getSubmissionId() . '-R' . $reviewAssignment->getRound();

			foreach ( $columns as $index => $junk )
				switch ( $index ) {
					case 'journal_title':
						$columns[ $index ] = $journalTitle;
						break;
					case 'journal_pissn':
						$columns[ $index ] = $journalPIssn;
						break;
					case 'journal_eissn':
						$columns[ $index ] = $journalEIssn;
						break;
					case 'reviewer_email':
						$columns[ $index ] = $reviewer->getEmail();
						break;
					case 'orcid':
						$columns[ $index ] = $reviewer->getOrcid();
						break;
					case 'manuscript':
						$columns[ $index ] = $manuscriptId;
						break;
					case 'revision':
					case 'open_pr':
					case 'doi':
					case 'empty':
					case 'instructions':
						$columns[ $index ] = null;
						break;
					case 'recommendation':
						if ( isset( $recommendations[ $row->$index ] ) ) {
							$columns[ $index ] = ( ! isset( $row->$index ) ) ? __( 'common.none' ) : __( $recommendations[ $row->$index ] );
						} else {
							$columns[ $index ] = '';
						}
						break;

					case 'date_accepted':
						if ( empty( $reviewAssignment->getDateConfirmed() ) ) {
							$columns[ $index ] = "";
						} else {
							$columns[ $index ] = DateTime::createFromFormat( 'Y-m-d H:i:s', $reviewAssignment->getDateConfirmed() )->format( 'Y-m-d H:i:s' );
						}
						break;

					case 'comment_length':
						if ( isset( $comments[ $row->submission_id ][ $row->reviewer_id ][ $reviewId ] ) ) {
							$columns[ $index ] = strlen( $comments[ $row->submission_id ][ $row->reviewer_id ][ $reviewId ] ) - 7; //ojs adds <p></p> at the beginning and end of the comment so -7.
						} else {
							$columns[ $index ] = 0;
						}
						break;
					default:
						$columns[ $index ] = $row->$index;
				}
			fputcsv( $fp, $columns );
		}
		fclose( $fp );
	}

}
